//=============================================================================
// KNSMZ_EquipJob.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc ver.1.0.0 装備変更で職業を変更します
 * @author 莞爾の草
 * @url https://kanjinokusargss3.hatenablog.com/
 * 
 * @param keepLevel
 * @text 職業間で経験値を維持
 * @desc ONの場合、職業変更時経験値が最も高い職業の値に変更されます。
 * @type boolean
 * @default false
 * 
 * @param keepHpMp
 * @text 装備画面終了時HP/MPを維持
 * @desc 職業が変わるとHP/MPが逐次丸め込まれるため、画面終了時開始時に保存した値に置き換える処理をします
 * @type boolean
 * @default true
 * 
 * @param considerOptimize
 * @text 「最強装備」で職業を維持
 * @desc ONの場合職業を指定した装備品が「最強装備」で変更されなくなります。
 * @type boolean
 * @default true
 * 
 * @help
 * アクター、武器、防具のメモ欄に下記のように記述することで
 * 装備変更時、職業を変更できます。
 * <KNS_Job: x>
 * （xに変更する職業IDを指定、1以上の自然数）
 * 
 * 上記記述の優先度は下記の通りです。
 * 1. 装備品(複数ある場合は*装備順が上のもの)
 * 2. データベース「アクター」メモ欄で指定された職業ID
 * 3. データベース「アクター」で指定された職業ID
 * 
 * *デフォルトの場合装備品は「剣、盾、頭……」の順で続きますが、
 *  剣と盾で別々の職業が指定されている場合は上記の順番が上の
 *  剣が優先されます。
 * 
 * ■更新履歴
 * ver. 1.0.0(2022-5-1)
 * - デモ版
 */

//========================================================
// new KNSMZ_EquipJob
//========================================================
const KNSMZ_EquipJob = {
	name: 'KNSMZ_EquipJob',
	reCheckNote: /<KNS_Job[:：]\s*(\d+)>/i,
	hasItemClass: function(item){
		if (item && this.reCheckNote.test(item.note)){
			return Math.floor(RegExp.$1);
		}else{
			return 0;
		}
	}
};


(function(){
	this.param = PluginManager.parameters(this.name);
	const extractBool = function(name){
		this[name] = this.param[name] == 'true';
	}.bind(this);
	extractBool('keepLevel');
	extractBool('keepHpMp');
	extractBool('considerOptimize');
}).call(KNSMZ_EquipJob);

(function(){
	//========================================================
	// alias Game_Actor
	//========================================================
	const _Game_Actor_setup = Game_Actor.prototype.setup;
	Game_Actor.prototype.setup = function(actorId) {
		_Game_Actor_setup.apply(this, arguments);
		this._knsIsTempActor = false;
		this.knsRefreshEquipJob();
		this.refresh();
	};

	const _Game_Actor_releaseUnequippableItems = Game_Actor.prototype.releaseUnequippableItems;
	Game_Actor.prototype.releaseUnequippableItems = function(forcing){
		if (this._knsIsTempActor){
			forcing = true;
		}
		_Game_Actor_releaseUnequippableItems.call(this, forcing);
	};

	const _Game_Actor_refresh = Game_Actor.prototype.refresh;
	Game_Actor.prototype.refresh = function() {
		this.releaseUnequippableItems(false);
		this.knsRefreshEquipJob();
		_Game_Actor_refresh.apply(this, arguments);
	};

	const _Game_Actor_optimizeEquipments = Game_Actor.prototype.optimizeEquipments;
	Game_Actor.prototype.optimizeEquipments = function(){
		if (KNSMZ_EquipJob.considerOptimize == true){
			const oldOk = Game_Actor.prototype.isEquipChangeOk;
			Game_Actor.prototype.isEquipChangeOk = function(slotId){
				return	KNSMZ_EquipJob.hasItemClass(this.equips()[slotId]) == 0 &&
						oldOk.apply(this, arguments);
			}
			_Game_Actor_optimizeEquipments.apply(this, arguments);
			Game_Actor.prototype.isEquipChangeOk = oldOk;
		}else{
			_Game_Actor_optimizeEquipments.apply(this, arguments);
		}
	};

	// new
	Game_Actor.prototype.knsRefreshEquipJob = function(){
		for (let i = 0; i < 2; i++){
			const newClassId = this.knsGetEquipJob();
			if (!this.isClass($dataClasses[newClassId])){
				if (KNSMZ_EquipJob.keepLevel == true){
					let maxExp = -1;
					Object.keys(this._exp).forEach(function(id){
						const cur = this._exp[id];
						if (maxExp <= cur){ maxExp = cur; }
					}, this);
					if (maxExp >= 0){
						const oldRefresh = Game_Actor.prototype.refresh;
						Game_Actor.prototype.refresh = _Game_Actor_refresh;
						this.changeExp(maxExp, false);
						Game_Actor.prototype.refresh = oldRefresh;
					}
					this.changeClass(newClassId, true);
				}else{
					this.changeClass(newClassId, false);
				}
			}
		}
	}

	Game_Actor.prototype.knsGetEquipJob = function(){
		let classId = 0;
		const equips = this.equips();
		for (let i = 0; i < equips.length; i++){
			const cur = KNSMZ_EquipJob.hasItemClass(equips[i]);
			if (cur != 0){ classId = cur; break; }
		}
		if (classId != 0){
			return classId;
		}else{
			const actor = this.actor();
			return	KNSMZ_EquipJob.hasItemClass(actor) || actor.classId;
		}
	}

	//========================================================
	// alias Window_EquipItem
	//========================================================
	const _Window_EquipItem_updateHelp = Window_EquipItem.prototype.updateHelp;
	Window_EquipItem.prototype.updateHelp = function() {
		const old = JsonEx.makeDeepCopy;
		JsonEx.makeDeepCopy = function (obj){
			const copy = old.apply(this, arguments);
			if (copy){ copy._knsIsTempActor = true; }
			return copy;
		}
		_Window_EquipItem_updateHelp.call(this);
		JsonEx.makeDeepCopy = old;
	};

	//========================================================
	// alias Scene_Equip
	//========================================================
	const _Scene_Equip_create = Scene_Equip.prototype.create;
	Scene_Equip.prototype.create = function(){
		if (KNSMZ_EquipJob.keepHpMp == true){
			this._knsActorInfos = {};
			$gameParty.members().forEach(function(actor){
				this._knsActorInfos[actor.actorId()] = [actor.hp, actor.mp];
			}, this);
		}
		_Scene_Equip_create.apply(this, arguments);
	};
	
	const _Scene_Equip_terminate = Scene_Equip.prototype.create;
	Scene_Equip.prototype.terminate = function(){
		if (KNSMZ_EquipJob.keepHpMp == true){
			Object.keys(this._knsActorInfos).forEach(function(id){
				const info = this._knsActorInfos[id];
				if (info){
					const actor = $gameActors.actor(id);
					actor.setHp(info[0]);
					actor.setMp(info[1]);
				}
			}, this);
		}
		_Scene_Equip_terminate.apply(this, arguments);
	};
})();